
import sys
import time
import numpy as np
import adi
#from test.eeprom import read_fru_eeprom
import matplotlib.pyplot as plt
import math
from scipy.fft import rfft, rfftfreq
from scipy import signal
from random import uniform


# user inputs
Kp = 1
Ki = 0.2
Kd = 0.01

def set_Kp(kp):
    """Write to axi register to set Kp"""
    hdl_dut_write_channel.axi4_lite_register_write(0x10c, kp*2**10)

def set_Ki(ki):
    """Write to axi register to set Ki"""
    hdl_dut_write_channel.axi4_lite_register_write(0x110, ki*2**10)

def set_Kd(kd):
    """Write to axi register to set Kd"""
    hdl_dut_write_channel.axi4_lite_register_write(0x114, kd*2**10)

def enable_plant(enp):
    """Write to axi register to enable or bypass plant in feedback loop"""
    hdl_dut_write_channel.axi4_lite_register_write(0x118, enp)

def get_kp():
    """Read axi register that contains Kp and convert to decimal"""
    kp_regRead = hdl_dut_read_channel.axi4_lite_register_read(0x10c)
    return int(kp_regRead, 16)/2**10

def get_ki():
    """Read axi register that contains Ki and convert to decimal"""
    ki_regRead = hdl_dut_read_channel.axi4_lite_register_read(0x110)
    return int(ki_regRead, 16)/2**10

def get_kd():
    """Read axi register that contains Kd and convert to decimal"""
    kd_regRead = hdl_dut_read_channel.axi4_lite_register_read(0x114)
    return int(kd_regRead, 16)/2**10

adaq23876_vref = 2.048
adaq23876_gain = 0.37 / 4.07
# Optionally passs URI as command line argument,
# else use default ip:analog.local
my_uri = sys.argv[1] if len(sys.argv) >= 2 else "ip:169.254.47.101"
print("uri: " + str(my_uri))


class adaq23876(adi.ltc2387):
    _rx_channel_names = ["voltage1", "voltage2"]
    _rx_data_type = np.int16


# device connections
adaq23876_adc = adaq23876(my_uri)
ad3552r_0 = adi.ad3552r(uri=my_uri, device_name="axi-ad3552r-0")
ad3552r_1 = adi.ad3552r(uri=my_uri, device_name="axi-ad3552r-1")
voltage_monitor = adi.ad7291(uri=my_uri)
gpio_controller = adi.one_bit_adc_dac(uri=my_uri, name="one-bit-adc-dac")

print("#############################################")

# gpio values setup
gpio_controller.gpio_pad_adc3 = 1
gpio_controller.gpio_pad_adc2 = 1
gpio_controller.gpio_pad_adc1 = 1
gpio_controller.gpio_pad_adc0 = 1

gpio_controller.gpio_gpio0_vio = 1
gpio_controller.gpio_gpio1_vio = 1
gpio_controller.gpio_gpio2_vio = 1
gpio_controller.gpio_gpio3_vio = 1

gpio_controller.gpio_gpio6_vio = 1
gpio_controller.gpio_gpio7_vio = 1

print("GPIO4_VIO state is:", gpio_controller.gpio_gpio4_vio)
print("GPIO5_VIO state is:", gpio_controller.gpio_gpio5_vio)

# voltage measurements
print("Voltage monitor values:")
print("Temperature: ", voltage_monitor.temp0(), " C")
print("Channel 0: ", voltage_monitor.voltage0(), " millivolts")
print("Channel 1: ", voltage_monitor.voltage1(), " millivolts")
print("Channel 2: ", voltage_monitor.voltage2(), " millivolts")
print("Channel 3: ", voltage_monitor.voltage3(), " millivolts")
print("Channel 4: ", voltage_monitor.voltage4(), " millivolts")
print("Channel 5: ", voltage_monitor.voltage5(), " millivolts")
print("Channel 6: ", voltage_monitor.voltage6(), " millivolts")
print("Channel 7: ", voltage_monitor.voltage7(), " millivolts")

# device configurations

hdl_dut_write_channel = adi.mwpicore(uri=my_uri, device_name="mwipcore0:mmwr-channel0")
hdl_dut_read_channel = adi.mwpicore(uri=my_uri, device_name="mwipcore0:mmrd-channel1")

adaq23876_adc.rx_buffer_size = 20000
print("Buffer size is ", adaq23876_adc.rx_buffer_size)
adaq23876_adc.sampling_frequency = 15000000

ad3552r_0.tx_enabled_channels = [0, 1]
ad3552r_1.tx_enabled_channels = [0, 1]
ad3552r_0.tx_cyclic_buffer = True
ad3552r_1.tx_cyclic_buffer = True


N = int(15000000/500)
samples = np.ones(N)
#samples = samples*6553
samples[0:int(N/2)] = samples[0:int(N/2)]*-6553
samples[int(N/2)+1:-1] = samples[int(N/2)+1:-1]*6553
samples = np.uint16(samples)

print("Sampling rate is:", ad3552r_0.sample_rate)

# available options: "0/2.5V", "0/5V", "0/10V", "-5/+5V", "-10/+10V"
ad3552r_0.output_range = "-10/+10V"
ad3552r_1.output_range = "-10/+10V"

# available options:"adc_input", "dma_input", "ramp_input"
ad3552r_0.input_source = "adc_input"
ad3552r_1.input_source = "dma_input"

print("input_source:dac0:", ad3552r_0.input_source)
print("input_source:dac1:", ad3552r_1.input_source)

# DAC 1 has to be updated and started first and then DAC0 in order to have syncronized data between devices 
ad3552r_1.tx([samples,samples])
ad3552r_0.tx([samples,samples])

# available options:"start_stream_synced", "start_stream", "stop_stream"
ad3552r_1.stream_status = "start_stream_synced"
ad3552r_0.stream_status = "start_stream_synced"
print("#############################################")

# Set axi registers
if hdl_dut_write_channel.check_matlab_ip() : 
    set_Kp(Kp)
    set_Ki(Ki)
    set_Kd(Kd)

    print("PID controller Kp given as: ", Kp)
    print("PID controller Ki given as: ", Ki)
    print("PID controller Kd given as: ", Kd)

    kp_readback = get_kp()
    ki_readback = get_ki()
    kd_readback = get_kd()
    print("Register value for kp:", int(kp_readback*2**10), "decimal value: ",kp_readback)
    print("Register value for ki:", int(ki_readback*2**10), "decimal value: ",ki_readback)
    print("Register value for kd:", int(kd_readback*2**10), "decimal value: ",kd_readback)
    
# Catpure data and plot

x = np.arange(0, adaq23876_adc.rx_buffer_size)
data = adaq23876_adc.rx()
voltage_1 = ( data[0]  * -adaq23876_vref )  / (adaq23876_gain * 2 ** 16)
voltage_2 = ( data[1]  * -adaq23876_vref )  / (adaq23876_gain * 2 ** 16)


fig, (setPoint, feedback) = plt.subplots(2, 1)
fig.suptitle("PID")
setPoint.plot(x, voltage_2[0 : adaq23876_adc.rx_buffer_size])
feedback.plot(x, voltage_1[0 : adaq23876_adc.rx_buffer_size])
setPoint.set_ylabel("Set Point \nCh 2 [V]")
feedback.set_ylabel("Feedback \nCh 1[V]")
plt.show()

# stop stream and destroy buffers 

ad3552r_1.stream_status = "stop_stream"
ad3552r_0.stream_status = "stop_stream"

ad3552r_1.tx_destroy_buffer()
ad3552r_0.tx_destroy_buffer()
adaq23876_adc.rx_destroy_buffer()

