
import sys
import time
import numpy as np
import adi
#from test.eeprom import read_fru_eeprom
import matplotlib.pyplot as plt
import math
from scipy.fft import rfft, rfftfreq
from scipy import signal


# user inputs
external_signals = 0 # 0 to use DMA generated as input in loopback mode, 1 to connect an external input to ADC and view output externally

def find_fft_max(frequencies, magnitudes):
    """Return the largest magnitude of the FFT with it's associated frequency and index in the input arrays"""
    peak_index = np.argmax(magnitudes)
    fund_frequency = frequencies[peak_index]
    peak_mag = max(magnitudes)
    return fund_frequency, peak_mag, peak_index

def snr_calc(frequencies, magnitude):
    """Calculate SNR from the FFT, return in dB"""
    fund_frequency, peak_mag, peak_index = find_fft_max(frequencies, magnitude)
    signal_start = peak_index-3
    signal_end = peak_index+4
    signal_bits = magnitude[signal_start:signal_end]
    signal_s = [f**2 for f in signal_bits]
    signal_vss = sum(signal_s)

    noise_bits = np.concatenate([magnitude[4:signal_start], magnitude[signal_end:]])
    noise_s = [f**2 for f in noise_bits]
    noise_vss = sum(noise_s)

    return 10*math.log10(signal_vss/noise_vss)

adaq23876_vref = 2.048
adaq23876_gain = 0.37 / 4.07
# Optionally passs URI as command line argument,
# else use default ip:analog.local
my_uri = sys.argv[1] if len(sys.argv) >= 2 else "ip:10.48.65.187"
print("uri: " + str(my_uri))


class adaq23876(adi.ltc2387):
    _rx_channel_names = ["voltage0", "voltage3"]
    _rx_data_type = np.int16


# device connections
adaq23876_adc = adaq23876(my_uri)
ad3552r_0 = adi.ad3552r(uri=my_uri, device_name="axi-ad3552r-0")
ad3552r_1 = adi.ad3552r(uri=my_uri, device_name="axi-ad3552r-1")
voltage_monitor = adi.ad7291(uri=my_uri)
gpio_controller = adi.one_bit_adc_dac(uri=my_uri, name="one-bit-adc-dac")

print("#############################################")

# gpio values setup
gpio_controller.gpio_pad_adc3 = 1
gpio_controller.gpio_pad_adc2 = 1
gpio_controller.gpio_pad_adc1 = 1
gpio_controller.gpio_pad_adc0 = 1

gpio_controller.gpio_gpio0_vio = 1
gpio_controller.gpio_gpio1_vio = 1
gpio_controller.gpio_gpio2_vio = 1
gpio_controller.gpio_gpio3_vio = 1

gpio_controller.gpio_gpio6_vio = 1
gpio_controller.gpio_gpio7_vio = 1

print("GPIO4_VIO state is:", gpio_controller.gpio_gpio4_vio)
print("GPIO5_VIO state is:", gpio_controller.gpio_gpio5_vio)

# voltage measurements
print("Voltage monitor values:")
print("Temperature: ", voltage_monitor.temp0(), " C")
print("Channel 0: ", voltage_monitor.voltage0(), " millivolts")
print("Channel 1: ", voltage_monitor.voltage1(), " millivolts")
print("Channel 2: ", voltage_monitor.voltage2(), " millivolts")
print("Channel 3: ", voltage_monitor.voltage3(), " millivolts")
print("Channel 4: ", voltage_monitor.voltage4(), " millivolts")
print("Channel 5: ", voltage_monitor.voltage5(), " millivolts")
print("Channel 6: ", voltage_monitor.voltage6(), " millivolts")
print("Channel 7: ", voltage_monitor.voltage7(), " millivolts")

# device configurations

hdl_dut_write_channel = adi.mwpicore(uri=my_uri, device_name="mwipcore0:mmwr-channel0")
hdl_dut_read_channel = adi.mwpicore(uri=my_uri, device_name="mwipcore0:mmrd-channel1")

adaq23876_adc.rx_buffer_size = 4096
print("Buffer size is ", adaq23876_adc.rx_buffer_size)
adaq23876_adc.sampling_frequency = 15000000

ad3552r_0.tx_enabled_channels = [0, 1]
ad3552r_1.tx_enabled_channels = [0, 1]
ad3552r_0.tx_cyclic_buffer = True
ad3552r_1.tx_cyclic_buffer = True


# signal generation
fs = int(ad3552r_0.sample_rate)
# Signal frequency
fc = 10000
# Number of samples
N = int(fs / fc)
# Period
ts = 1 / float(fs)
# Time array
t = np.arange(0, N * ts, ts)
# Sine generation
samples = np.sin(2 * np.pi * t * fc)
# Amplitude (full_scale / 4)
samples *= (2 ** 14) - 1
# Offset (full_scale / 2)
samples += 2 ** 15
noise = np.random.normal(0,3000,N)
sineNoise_fc = 800000
sineNoise = np.sin(2 * np.pi * t * sineNoise_fc)
sineNoise *= (2**12)
samples = [s+n+n_s for s,n,n_s in zip(samples, noise, sineNoise)]
# conversion to unsigned int and offset binary
samples = np.uint16(samples)
samples = np.bitwise_xor(32768, samples)


print("Sampling rate is:", ad3552r_0.sample_rate)

# available options: "0/2.5V", "0/5V", "0/10V", "-5/+5V", "-10/+10V"
ad3552r_0.output_range = "-10/+10V"
ad3552r_1.output_range = "-10/+10V"

# available options:"adc_input", "dma_input", "ramp_input"
ad3552r_0.input_source = "adc_input"
ad3552r_1.input_source = "dma_input"

print("input_source:dac0:", ad3552r_0.input_source)
print("input_source:dac1:", ad3552r_1.input_source)

# DAC 1 has to be updated and started first and then DAC0 in order to have syncronized data between devices 
ad3552r_1.tx([samples,samples])
ad3552r_0.tx([samples,samples])

# available options:"start_stream_synced", "start_stream", "stop_stream"
ad3552r_1.stream_status = "start_stream_synced"
ad3552r_0.stream_status = "start_stream_synced"
print("#############################################")


if not external_signals:
    # Catpure data and plot
    x = np.arange(0, adaq23876_adc.rx_buffer_size)
    data = adaq23876_adc.rx()
    voltage_0 = ( data[0]  * -adaq23876_vref )  / (adaq23876_gain * 2 ** 16)
    voltage_3 = ( data[1]  * -adaq23876_vref )  / (adaq23876_gain * 2 ** 16)

    # Calculate FFTs
    window = signal.windows.hann(adaq23876_adc.rx_buffer_size)
    fft_freqs = rfftfreq(adaq23876_adc.rx_buffer_size, 1 / (adaq23876_adc.sampling_frequency / len(adaq23876_adc._rx_channel_names)))
    fft_firIn = np.abs(rfft(voltage_3[0:adaq23876_adc.rx_buffer_size]*window))/(adaq23876_adc.rx_buffer_size/2)
    fft_firOut = (np.abs(rfft(voltage_0[0:adaq23876_adc.rx_buffer_size]*window)))/(adaq23876_adc.rx_buffer_size/2)

    fig, (firIn, firOut, fftIn, fftOut) = plt.subplots(4, 1)
    fig.suptitle("FIR filter input and outputs")
    firIn.plot(x, voltage_3[0 : adaq23876_adc.rx_buffer_size])
    firOut.plot(x, voltage_0[0 : adaq23876_adc.rx_buffer_size])
    firIn.set_ylabel("FIR In \n Ch 3 [V]")
    firOut.set_ylabel("FIR Out \n Ch 0 [V]")
    firOut.set_xlabel("Samples")
    fftIn.plot(fft_freqs, [20*math.log10(mag) for mag in fft_firIn])
    fftOut.plot(fft_freqs, [20*math.log10(mag) for mag in fft_firOut])
    fftIn.set_ylabel("FIR In FFT [dB]")
    fftOut.set_ylabel("FIR Out FFT [dB]")
    fftOut.set_xlabel("Frequency")
    plt.subplots_adjust(top=0.9, hspace=0.351)
    plt.show()

    snr0 = snr_calc(fft_freqs, fft_firIn)
    snr_filtered = snr_calc(fft_freqs, fft_firOut)
    print("SNR of unfiltered signal: ", snr0, "dB")
    print("SNR of filtered signal: ", snr_filtered, "dB")

    max_frequency, peak_mag, peak_index = find_fft_max(fft_freqs[100:], fft_firIn[100:])
    peak_index = peak_index+100
    signalMag_preFilt = sum(fft_firIn[peak_index-3:peak_index+4])
    signalMag_postFilt = sum(fft_firOut[peak_index-3:peak_index+4])
    signalAtten = signalMag_postFilt/signalMag_preFilt
    improvedFreqEst = 0
    for idx in range(peak_index-3, peak_index+4):
        improvedFreqEst += fft_firOut[idx] * idx * adaq23876_adc.sampling_frequency / adaq23876_adc.rx_buffer_size /2
    improvedFreqEst = improvedFreqEst/sum(fft_firOut[peak_index-3:peak_index+4])
    print("The signal at ", round(improvedFreqEst), "Hz was attenuated by ", -20*math.log10(signalAtten), "dB")
else:
    print("ADC data not captured, DAC signal should be viewed externally")
    input("Press enter to stop stream and destroy buffer...")


# stop stream and destroy buffers 

ad3552r_1.stream_status = "stop_stream"
ad3552r_0.stream_status = "stop_stream"

ad3552r_1.tx_destroy_buffer()
ad3552r_0.tx_destroy_buffer()
adaq23876_adc.rx_destroy_buffer()

