
import sys
import time
import numpy as np
import adi
#from test.eeprom import read_fru_eeprom
import matplotlib.pyplot as plt
import math
from scipy.fft import rfft, rfftfreq
from scipy import signal

# user inputs
dds_freq = 2000
dds_amp = 2**10
dds_phase_shift = 0
dma_freq = 3000
dma_amp = 2**7
external_signals = 0

def set_freq(freq):
    """Write to axi register to set frequency of DDS output"""
    hdl_dut_write_channel.axi4_lite_register_write(0x100, freq)

def set_amp(amp):
    """Write to axi register to set amplitude of DDS output"""
    hdl_dut_write_channel.axi4_lite_register_write(0x104, amp)

def set_phase(phase):
    """Write to axi register to set phase shift of DDS output"""
    hdl_dut_write_channel.axi4_lite_register_write(0x108, phase)

def find_fft_max(frequencies, magnitude):
    """Return the largest magnitude of the FFT with it's associated frequency and index in the input arrays"""
    peak_index = np.argmax(magnitude)
    fund_frequency = frequencies[peak_index]
    peak_mag = max(magnitude)
    return fund_frequency, peak_mag, peak_index

adaq23876_vref = 2.048
adaq23876_gain = 0.37 / 4.07
# Optionally passs URI as command line argument,
# else use default ip:analog.local
my_uri = sys.argv[1] if len(sys.argv) >= 2 else "ip:10.48.65.187"
print("uri: " + str(my_uri))


class adaq23876(adi.ltc2387):
    _rx_channel_names = ["voltage0", "voltage2", "voltage3"]
    _rx_data_type = np.int16


# device connections
adaq23876_adc = adaq23876(my_uri)
ad3552r_0 = adi.ad3552r(uri=my_uri, device_name="axi-ad3552r-0")
ad3552r_1 = adi.ad3552r(uri=my_uri, device_name="axi-ad3552r-1")
voltage_monitor = adi.ad7291(uri=my_uri)
gpio_controller = adi.one_bit_adc_dac(uri=my_uri, name="one-bit-adc-dac")

print("#############################################")

# gpio values setup
gpio_controller.gpio_pad_adc3 = 1
gpio_controller.gpio_pad_adc2 = 1
gpio_controller.gpio_pad_adc1 = 1
gpio_controller.gpio_pad_adc0 = 1

gpio_controller.gpio_gpio0_vio = 1
gpio_controller.gpio_gpio1_vio = 1
gpio_controller.gpio_gpio2_vio = 1
gpio_controller.gpio_gpio3_vio = 1

gpio_controller.gpio_gpio6_vio = 1
gpio_controller.gpio_gpio7_vio = 1

print("GPIO4_VIO state is:", gpio_controller.gpio_gpio4_vio)
print("GPIO5_VIO state is:", gpio_controller.gpio_gpio5_vio)

# voltage measurements
print("Voltage monitor values:")
print("Temperature: ", voltage_monitor.temp0(), " C")
print("Channel 0: ", voltage_monitor.voltage0(), " millivolts")
print("Channel 1: ", voltage_monitor.voltage1(), " millivolts")
print("Channel 2: ", voltage_monitor.voltage2(), " millivolts")
print("Channel 3: ", voltage_monitor.voltage3(), " millivolts")
print("Channel 4: ", voltage_monitor.voltage4(), " millivolts")
print("Channel 5: ", voltage_monitor.voltage5(), " millivolts")
print("Channel 6: ", voltage_monitor.voltage6(), " millivolts")
print("Channel 7: ", voltage_monitor.voltage7(), " millivolts")

# device configurations

hdl_dut_write_channel = adi.mwpicore(uri=my_uri, device_name="mwipcore0:mmwr-channel0")
hdl_dut_read_channel = adi.mwpicore(uri=my_uri, device_name="mwipcore0:mmrd-channel1")

adaq23876_adc.rx_buffer_size = 150000
print("Buffer size is ", adaq23876_adc.rx_buffer_size)
adaq23876_adc.sampling_frequency = 15000000

ad3552r_0.tx_enabled_channels = [0, 1]
ad3552r_1.tx_enabled_channels = [0, 1]
ad3552r_0.tx_cyclic_buffer = True
ad3552r_1.tx_cyclic_buffer = True


# signal generation
fs = int(ad3552r_0.sample_rate)
# Signal frequency
fc = dma_freq
# Number of samples
N = int(fs / fc)
# Period
ts = 1 / float(fs)
# Time array
t = np.arange(0, N * ts, ts)
# Sine generation
samples = np.sin(2 * np.pi * t * fc)
# Amplitude (full_scale / 4)
samples *= dma_amp
# Offset (full_scale / 2)
samples += 2 ** 15
# conversion to unsigned int and offset binary
samples = np.uint16(samples)
samples = np.bitwise_xor(32768, samples)

print("Sampling rate is:", ad3552r_0.sample_rate)

# available options: "0/2.5V", "0/5V", "0/10V", "-5/+5V", "-10/+10V"
ad3552r_0.output_range = "-10/+10V"
ad3552r_1.output_range = "-10/+10V"
if ad3552r_0.output_range == "-10/+10V":
    amp_scale = 20
elif ad3552r_0.output_range == "0/2.5V":
    amp_scale = 2.5
elif ad3552r_0.output_range == "0/5V":
    amp_scale = 5
else:
    amp_scale = 10
amp_scale = amp_scale / 2**16

# available options:"adc_input", "dma_input", "ramp_input"
ad3552r_0.input_source = "dma_input"
ad3552r_1.input_source = "adc_input"

print("input_source:dac0:", ad3552r_0.input_source)
print("input_source:dac1:", ad3552r_1.input_source)

# DAC 1 has to be updated and started first and then DAC0 in order to have syncronized data between devices 
ad3552r_1.tx([samples,samples])
ad3552r_0.tx([samples,samples])

# available options:"start_stream_synced", "start_stream", "stop_stream"
ad3552r_1.stream_status = "start_stream_synced"
ad3552r_0.stream_status = "start_stream_synced"
print("#############################################")

# Set axi registers
if hdl_dut_write_channel.check_matlab_ip() : 
    set_freq(dds_freq)
    set_amp(dds_amp)
    set_phase(dds_phase_shift)
    print("DDS frequency set to ", dds_freq, "Hz")
    print("DDS amplitude set to ", dds_amp*amp_scale, "V")
    print("DDS phase shift set to ", dds_phase_shift, "degrees")
print("DMA frequency set to ", dma_freq, "Hz")
print("DMA amplitude set to ", dma_amp*amp_scale, "V")
    
if not external_signals:
    # Catpure data and plot
    x = np.arange(0, adaq23876_adc.rx_buffer_size)
    data = adaq23876_adc.rx()
    voltage_0 = ( data[0]  * -adaq23876_vref )  / (adaq23876_gain * 2 ** 16)
    voltage_2 = ( data[1]  * -adaq23876_vref )  / (adaq23876_gain * 2 ** 16)
    voltage_3 = ( data[2]  * -adaq23876_vref )  / (adaq23876_gain * 2 ** 16)
    # Perform FFT on channel
    window = signal.windows.hann(adaq23876_adc.rx_buffer_size)
    fft_out = rfft(voltage_2[0:adaq23876_adc.rx_buffer_size]*window)
    fft_out_norm = np.abs(fft_out)/(adaq23876_adc.rx_buffer_size/2)
    fft_freqs = rfftfreq(adaq23876_adc.rx_buffer_size, 1 / (adaq23876_adc.sampling_frequency / len(adaq23876_adc._rx_channel_names)))

    fig, (ch3, ch0, ch2, fft) = plt.subplots(4, 1)
    fig.suptitle("Mixer Inputs and  Outputs")
    ch0.plot(x, voltage_0[0 : adaq23876_adc.rx_buffer_size])
    ch2.plot(x, voltage_2[0 : adaq23876_adc.rx_buffer_size])
    ch3.plot(x, voltage_3[0 : adaq23876_adc.rx_buffer_size])
    fft.plot(fft_freqs, [20*math.log10(mag) for mag in fft_out_norm])
    ch3.set_ylabel("DDS Out \n Ch 3 [V]")
    ch0.set_ylabel("Analog In \n Ch 0 [V]")
    ch2.set_ylabel("Mixer Out \n Ch 2 [V]")
    fft.set_ylabel("Mixer Out FFT [dB]")
    ch2.set_xlabel("Samples")
    fft.set_xlabel("Frequency")
    fft.set_xlim([0, (dds_freq+dma_freq)*3])
    plt.subplots_adjust(top=0.9, hspace=0.351)
    plt.show()

    peak_freq, peak_mag, pdx = find_fft_max(fft_freqs, fft_out_norm)
    signal_power = 20*math.log10(sum(fft_out_norm[pdx-3:pdx+4]))
    print("The mixer output's largest frequency component is at ", peak_freq/1000, "kHz, with estimated signal power ", round(signal_power,2), " dB")
else:
    print("ADC data not captured, DAC signal should be viewed externally")
    input("Press enter to stop stream and destroy buffer...")

# stop stream and destroy buffers 

ad3552r_1.stream_status = "stop_stream"
ad3552r_0.stream_status = "stop_stream"

ad3552r_1.tx_destroy_buffer()
ad3552r_0.tx_destroy_buffer()
adaq23876_adc.rx_destroy_buffer()